<?php
/**
 * BankInstallmentLoanController - Aylik Es Taksitli Kredi Yonetimi
 */
class BankInstallmentLoanController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * Taksitli kredi listesi
     */
    public function index() {
        $loans = $this->db->select("
            SELECT
                l.id,
                l.bank_name,
                l.loan_amount,
                l.monthly_payment,
                l.total_installments,
                l.remaining_installments,
                l.start_date,
                l.next_payment_date,
                DAY(l.next_payment_date) as payment_day,
                l.status,
                l.description,
                (
                    SELECT p.id
                    FROM bank_installment_payments p
                    WHERE p.loan_id = l.id AND p.status = 'BEKLEMEDE'
                    ORDER BY p.payment_number ASC
                    LIMIT 1
                ) as next_payment_id,
                (
                    SELECT p.payment_number
                    FROM bank_installment_payments p
                    WHERE p.loan_id = l.id AND p.status = 'BEKLEMEDE'
                    ORDER BY p.payment_number ASC
                    LIMIT 1
                ) as next_payment_number
            FROM bank_installment_loans l
            ORDER BY l.next_payment_date ASC, l.bank_name ASC
        ");

        $summary = $this->db->selectOne("
            SELECT
                COUNT(*) as loan_count,
                SUM(loan_amount) as total_loan_amount,
                SUM(monthly_payment) as total_monthly_payment
            FROM bank_installment_loans
            WHERE status <> 'TAMAMLANDI'
        ");

        $payments = $this->db->select("
            SELECT
                p.id,
                p.loan_id,
                p.payment_number,
                p.payment_date,
                p.amount,
                p.status,
                p.paid_date
            FROM bank_installment_payments p
            ORDER BY p.loan_id ASC, p.payment_number ASC
        ");

        $paymentMap = [];
        foreach ($payments as $payment) {
            $loanId = (int)$payment['loan_id'];
            if (!isset($paymentMap[$loanId])) {
                $paymentMap[$loanId] = [];
            }
            $paymentMap[$loanId][] = $payment;
        }

        $auth = $this->auth;
        $pageTitle = 'Aylik Es Taksitli Krediler';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/installment-loans/index.php';
    }

    /**
     * Yeni taksitli kredi ekleme
     */
    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/installment-loans');
        }

        $bankName = trim($_POST['bank_name'] ?? '');
        $loanAmount = floatval($_POST['loan_amount'] ?? 0);
        $monthlyPayment = floatval($_POST['monthly_payment'] ?? 0);
        $paymentDay = intval($_POST['payment_day'] ?? 0);
        $totalInstallments = intval($_POST['total_installments'] ?? 0);
        $startDate = $_POST['start_date'] ?? date('Y-m-d');
        $description = trim($_POST['description'] ?? '');

        if ($bankName === '' || $loanAmount <= 0 || $monthlyPayment <= 0 || $paymentDay < 1 || $paymentDay > 31 || $totalInstallments <= 0) {
            Helper::setFlash('danger', 'Banka, kredi tutari, aylik taksit, odeme gunu ve taksit sayisi zorunludur.');
            Helper::redirect('/banks/installment-loans');
        }

        $firstPaymentDate = $this->calculateFirstPaymentDate($startDate, $paymentDay);

        try {
            $this->db->beginTransaction();

            $loanId = $this->db->insert('bank_installment_loans', [
                'bank_name' => $bankName,
                'loan_amount' => $loanAmount,
                'monthly_payment' => $monthlyPayment,
                'total_installments' => $totalInstallments,
                'remaining_installments' => $totalInstallments,
                'start_date' => $startDate,
                'next_payment_date' => $firstPaymentDate,
                'status' => 'DEVAM EDİYOR',
                'description' => $description !== '' ? $description : null,
                'created_by' => $_SESSION['user_id'] ?? null
            ]);

            if (!$loanId) {
                throw new Exception('Kredi kaydi eklenemedi.');
            }

            for ($i = 1; $i <= $totalInstallments; $i++) {
                $paymentDate = date('Y-m-d', strtotime($firstPaymentDate . ' +' . ($i - 1) . ' month'));

                $inserted = $this->db->insert('bank_installment_payments', [
                    'loan_id' => $loanId,
                    'payment_number' => $i,
                    'payment_date' => $paymentDate,
                    'amount' => $monthlyPayment,
                    'status' => 'BEKLEMEDE'
                ]);

                if (!$inserted) {
                    throw new Exception('Taksit plani olusturulamadi.');
                }
            }

            $this->db->commit();
            Helper::setFlash('success', 'Taksitli kredi ve odeme plani olusturuldu.');
        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'Kayit sirasinda hata olustu: ' . $e->getMessage());
        }

        Helper::redirect('/banks/installment-loans');
    }

    /**
     * Taksit odendi olarak isaretle
     */
    public function markPaid($loanId, $paymentId) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/installment-loans');
        }

        $loan = $this->db->selectOne("SELECT * FROM bank_installment_loans WHERE id = ?", [$loanId]);
        $payment = $this->db->selectOne("SELECT * FROM bank_installment_payments WHERE id = ? AND loan_id = ?", [$paymentId, $loanId]);

        if (!$loan || !$payment) {
            Helper::setFlash('danger', 'Kredi veya taksit kaydi bulunamadi.');
            Helper::redirect('/banks/installment-loans');
        }

        if ($payment['status'] === 'ÖDENDİ') {
            Helper::setFlash('warning', 'Bu taksit zaten odenmis.');
            Helper::redirect('/banks/installment-loans');
        }

        try {
            $this->db->beginTransaction();

            $updated = $this->db->update('bank_installment_payments', [
                'status' => 'ÖDENDİ',
                'paid_date' => date('Y-m-d')
            ], 'id = ?', [$paymentId]);

            if ($updated === false) {
                throw new Exception('Taksit durumu guncellenemedi.');
            }

            $this->recalculateLoan($loanId);
            $this->db->commit();
            Helper::setFlash('success', 'Taksit odendi olarak isaretlendi.');
        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'Islem basarisiz: ' . $e->getMessage());
        }

        Helper::redirect('/banks/installment-loans');
    }

    /**
     * Odenmis taksidi tekrar beklemede yap
     */
    public function markPending($loanId, $paymentId) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/installment-loans');
        }

        $loan = $this->db->selectOne("SELECT * FROM bank_installment_loans WHERE id = ?", [$loanId]);
        $payment = $this->db->selectOne("SELECT * FROM bank_installment_payments WHERE id = ? AND loan_id = ?", [$paymentId, $loanId]);

        if (!$loan || !$payment) {
            Helper::setFlash('danger', 'Kredi veya taksit kaydi bulunamadi.');
            Helper::redirect('/banks/installment-loans');
        }

        if ($payment['status'] === 'BEKLEMEDE') {
            Helper::setFlash('warning', 'Bu taksit zaten beklemede.');
            Helper::redirect('/banks/installment-loans');
        }

        try {
            $this->db->beginTransaction();

            $updated = $this->db->update('bank_installment_payments', [
                'status' => 'BEKLEMEDE',
                'paid_date' => null
            ], 'id = ?', [$paymentId]);

            if ($updated === false) {
                throw new Exception('Taksit durumu guncellenemedi.');
            }

            $this->recalculateLoan($loanId);
            $this->db->commit();
            Helper::setFlash('success', 'Taksit tekrar beklemeye alindi.');
        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'Islem basarisiz: ' . $e->getMessage());
        }

        Helper::redirect('/banks/installment-loans');
    }

    /**
     * Kredi sil
     */
    public function delete($loanId) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/installment-loans');
        }

        $deleted = $this->db->delete('bank_installment_loans', 'id = ?', [$loanId]);
        if ($deleted) {
            Helper::setFlash('success', 'Kredi kaydi silindi.');
        } else {
            Helper::setFlash('danger', 'Kredi kaydi silinemedi.');
        }

        Helper::redirect('/banks/installment-loans');
    }

    /**
     * Kredi kaydi guncelle
     */
    public function update($loanId) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/installment-loans');
        }

        $loan = $this->db->selectOne("SELECT * FROM bank_installment_loans WHERE id = ?", [$loanId]);
        if (!$loan) {
            Helper::setFlash('danger', 'Kredi kaydi bulunamadi.');
            Helper::redirect('/banks/installment-loans');
        }

        $bankName = trim($_POST['bank_name'] ?? '');
        $loanAmount = floatval($_POST['loan_amount'] ?? 0);
        $monthlyPayment = floatval($_POST['monthly_payment'] ?? 0);
        $paymentDay = intval($_POST['payment_day'] ?? 0);
        $totalInstallments = intval($_POST['total_installments'] ?? 0);
        $startDate = $_POST['start_date'] ?? $loan['start_date'];
        $description = trim($_POST['description'] ?? '');

        if ($bankName === '' || $loanAmount <= 0 || $monthlyPayment <= 0 || $paymentDay < 1 || $paymentDay > 31 || $totalInstallments <= 0) {
            Helper::setFlash('danger', 'Banka, kredi tutari, aylik taksit, odeme gunu ve taksit sayisi zorunludur.');
            Helper::redirect('/banks/installment-loans');
        }

        $paidCountRow = $this->db->selectOne("
            SELECT COUNT(*) as paid_count
            FROM bank_installment_payments
            WHERE loan_id = ? AND status = 'ÖDENDİ'
        ", [$loanId]);
        $paidCount = intval($paidCountRow['paid_count'] ?? 0);

        if ($totalInstallments < $paidCount) {
            Helper::setFlash('danger', 'Toplam taksit sayisi, odenmis taksit sayisindan kucuk olamaz.');
            Helper::redirect('/banks/installment-loans');
        }

        $baseDate = $this->calculateFirstPaymentDate($startDate, $paymentDay);

        try {
            $this->db->beginTransaction();

            // Odenmis taksitleri koruyoruz, beklemedekileri silip yeniden uretiyoruz.
            $this->db->query("DELETE FROM bank_installment_payments WHERE loan_id = ? AND status = 'BEKLEMEDE'", [$loanId]);

            $remainingCount = $totalInstallments - $paidCount;
            for ($n = $paidCount + 1; $n <= $totalInstallments; $n++) {
                $paymentDate = date('Y-m-d', strtotime($baseDate . ' +' . ($n - 1) . ' month'));
                $inserted = $this->db->insert('bank_installment_payments', [
                    'loan_id' => $loanId,
                    'payment_number' => $n,
                    'payment_date' => $paymentDate,
                    'amount' => $monthlyPayment,
                    'status' => 'BEKLEMEDE'
                ]);
                if (!$inserted) {
                    throw new Exception('Yeni taksit plani olusturulamadi.');
                }
            }

            $nextPaymentDate = $remainingCount > 0
                ? date('Y-m-d', strtotime($baseDate . ' +' . $paidCount . ' month'))
                : date('Y-m-d');

            $status = $remainingCount > 0 ? 'DEVAM EDİYOR' : 'TAMAMLANDI';

            $updated = $this->db->update('bank_installment_loans', [
                'bank_name' => $bankName,
                'loan_amount' => $loanAmount,
                'monthly_payment' => $monthlyPayment,
                'total_installments' => $totalInstallments,
                'remaining_installments' => $remainingCount,
                'start_date' => $startDate,
                'next_payment_date' => $nextPaymentDate,
                'status' => $status,
                'description' => $description !== '' ? $description : null
            ], 'id = ?', [$loanId]);

            if ($updated === false) {
                throw new Exception('Kredi kaydi guncellenemedi.');
            }

            $this->recalculateLoan($loanId);
            $this->db->commit();
            Helper::setFlash('success', 'Kredi kaydi guncellendi.');
        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'Guncelleme sirasinda hata olustu: ' . $e->getMessage());
        }

        Helper::redirect('/banks/installment-loans');
    }

    private function recalculateLoan($loanId) {
        $agg = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN 1 ELSE 0 END) as pending_count
            FROM bank_installment_payments
            WHERE loan_id = ?
        ", [$loanId]);

        $nextPending = $this->db->selectOne("
            SELECT payment_date
            FROM bank_installment_payments
            WHERE loan_id = ? AND status = 'BEKLEMEDE'
            ORDER BY payment_number ASC
            LIMIT 1
        ", [$loanId]);

        $lastPayment = $this->db->selectOne("
            SELECT payment_date
            FROM bank_installment_payments
            WHERE loan_id = ?
            ORDER BY payment_number DESC
            LIMIT 1
        ", [$loanId]);

        $pendingCount = intval($agg['pending_count'] ?? 0);
        $status = $pendingCount > 0 ? 'DEVAM EDİYOR' : 'TAMAMLANDI';
        $nextPaymentDate = $pendingCount > 0
            ? ($nextPending['payment_date'] ?? date('Y-m-d'))
            : ($lastPayment['payment_date'] ?? date('Y-m-d'));

        $this->db->update('bank_installment_loans', [
            'remaining_installments' => $pendingCount,
            'status' => $status,
            'next_payment_date' => $nextPaymentDate
        ], 'id = ?', [$loanId]);
    }

    private function calculateFirstPaymentDate($startDate, $paymentDay) {
        $year = intval(date('Y', strtotime($startDate)));
        $month = intval(date('m', strtotime($startDate)));
        $day = intval(date('d', strtotime($startDate)));

        if ($day > $paymentDay) {
            $month++;
            if ($month > 12) {
                $month = 1;
                $year++;
            }
        }

        $maxDay = cal_days_in_month(CAL_GREGORIAN, $month, $year);
        $safeDay = min($paymentDay, $maxDay);

        return sprintf('%04d-%02d-%02d', $year, $month, $safeDay);
    }
}
