<?php
/**
 * ReceivedCheckController - Alınan Çek Takibi
 */
class ReceivedCheckController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    public function index() {
        $this->ensureChecksTable();

        $checks = $this->db->select("
            SELECT
                rc.id,
                rc.customer_name,
                rc.bank_name,
                rc.amount,
                rc.due_date,
                rc.check_number,
                rc.status,
                rc.description,
                rc.created_at
            FROM received_checks rc
            ORDER BY rc.due_date ASC, rc.created_at DESC
        ");

        $summary = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN amount ELSE 0 END) as pending_total,
                SUM(CASE WHEN status = 'TAHSIL_EDILDI' THEN amount ELSE 0 END) as collected_total,
                SUM(CASE WHEN status = 'CIRO_EDILDI' THEN amount ELSE 0 END) as endorsed_total
            FROM received_checks
        ");

        $auth = $this->auth;
        $pageTitle = 'Alınan Çekler';
        $currentPage = 'received-checks';

        require_once __DIR__ . '/../views/received-checks/index.php';
    }

    public function create() {
        $auth = $this->auth;
        $pageTitle = 'Yeni Alınan Çek';
        $currentPage = 'received-checks';

        require_once __DIR__ . '/../views/received-checks/create.php';
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/received-checks');
        }

        $this->ensureChecksTable();

        // Validate required fields
        $customerName = trim($_POST['customer_name'] ?? '');
        $amount = $_POST['amount'] ?? 0;
        $dueDate = $_POST['due_date'] ?? '';

        if (empty($customerName) || empty($amount) || empty($dueDate)) {
            Helper::setFlash('danger', 'Müşteri adı, tutar ve vade tarihi zorunludur.');
            Helper::redirect('/received-checks/create');
        }

        $data = [
            'customer_name' => $customerName,
            'bank_name' => trim($_POST['bank_name'] ?? '') ?: null,
            'amount' => floatval($amount),
            'due_date' => $dueDate,
            'check_number' => trim($_POST['check_number'] ?? '') ?: null,
            'status' => 'BEKLEMEDE',
            'description' => $_POST['description'] ?? null,
            'created_by' => $this->auth->userId()
        ];

        $id = $this->db->insert('received_checks', $data);

        if ($id) {
            Helper::setFlash('success', 'Alınan çek başarıyla kaydedildi.');
            Helper::redirect('/received-checks');
        } else {
            Helper::setFlash('danger', 'Çek kaydedilirken hata oluştu.');
            Helper::redirect('/received-checks/create');
        }
    }

    public function markCollected($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/received-checks');
        }

        $this->ensureChecksTable();

        $check = $this->db->selectOne("SELECT * FROM received_checks WHERE id = ?", [$id]);
        if (!$check) {
            Helper::setFlash('danger', 'Çek kaydı bulunamadı.');
            Helper::redirect('/received-checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'TAHSIL_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Çek tahsil edildi olarak işaretlendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/received-checks');
    }

    public function markPending($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/received-checks');
        }

        $this->ensureChecksTable();

        $check = $this->db->selectOne("SELECT * FROM received_checks WHERE id = ?", [$id]);
        if (!$check) {
            Helper::setFlash('danger', 'Çek kaydı bulunamadı.');
            Helper::redirect('/received-checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'BEKLEMEDE'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Çek beklemede olarak güncellendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/received-checks');
    }

    public function markEndorsed($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/received-checks');
        }

        $this->ensureChecksTable();

        $check = $this->db->selectOne("SELECT * FROM received_checks WHERE id = ?", [$id]);
        if (!$check) {
            Helper::setFlash('danger', 'Çek kaydı bulunamadı.');
            Helper::redirect('/received-checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'CIRO_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Çek ciro edildi olarak işaretlendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/received-checks');
    }

    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/received-checks');
        }

        $this->ensureChecksTable();

        $result = $this->db->delete('received_checks', 'id = ?', [$id]);

        if ($result) {
            Helper::setFlash('success', 'Çek kaydı başarıyla silindi.');
        } else {
            Helper::setFlash('danger', 'Çek kaydı silinemedi.');
        }

        Helper::redirect('/received-checks');
    }

    private function ensureChecksTable() {
        $this->db->query("
            CREATE TABLE IF NOT EXISTS received_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                customer_name VARCHAR(200) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                due_date DATE NOT NULL,
                check_number VARCHAR(100),
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE', 'CIRO_EDILDI') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_customer_name (customer_name),
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");
    }
}
