<?php
/**
 * SupplierController - Tedarikçi Yönetimi
 */

class SupplierController
{
    private $db;
    private $auth;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    /**
     * Tedarikçi listesi
     */
    public function index()
    {
        $suppliers = $this->db->select("
            SELECT
                s.id,
                s.supplier_name,
                s.created_at,
                COALESCE(SUM(
                    CASE
                        WHEN st.transaction_type IN ('purchase', 'adjustment') THEN st.amount
                        WHEN st.transaction_type IN ('payment', 'refund') THEN -st.amount
                        ELSE 0
                    END
                ), 0) as total_debt,
                COUNT(st.id) as transaction_count,
                MAX(st.transaction_date) as last_transaction_date
            FROM suppliers s
            LEFT JOIN supplier_transactions st ON s.id = st.supplier_id
            GROUP BY s.id, s.supplier_name, s.created_at
            ORDER BY s.supplier_name ASC
        ");

        // Calculate overall summary
        $summary = [
            'total_suppliers' => count($suppliers),
            'total_debt' => array_sum(array_column($suppliers, 'total_debt')),
            'suppliers_with_debt' => count(array_filter($suppliers, function ($s) {
                return $s['total_debt'] > 0;
            }))
        ];

        $auth = $this->auth;
        $pageTitle = 'Tedarikçiler';
        $currentPage = 'suppliers';

        require_once __DIR__ . '/../views/suppliers/index.php';
    }

    /**
     * Yeni tedarikçi formu
     */
    public function create()
    {
        $auth = $this->auth;
        $pageTitle = 'Yeni Tedarikçi';
        $currentPage = 'suppliers';

        require_once __DIR__ . '/../views/suppliers/create.php';
    }

    /**
     * Tedarikçi düzenleme
     */
    public function edit($id)
    {
        $supplier = $this->db->selectOne("SELECT * FROM suppliers WHERE id = ?", [$id]);

        if (!$supplier) {
            Helper::setFlash('danger', 'Tedarikçi bulunamadı.');
            Helper::redirect('/suppliers');
        }

        $auth = $this->auth;
        $pageTitle = 'Tedarikçi Düzenle';
        $currentPage = 'suppliers';

        require_once __DIR__ . '/../views/suppliers/edit.php';
    }

    /**
     * Tedarikçi detay - İşlem geçmişi ile birlikte
     */
    public function view($id)
    {
        $supplier = $this->db->selectOne("SELECT * FROM suppliers WHERE id = ?", [$id]);

        if (!$supplier) {
            Helper::setFlash('danger', 'Tedarikçi bulunamadı.');
            Helper::redirect('/suppliers');
        }

        $this->ensureChecksTable();

        // Fetch all transactions for this supplier
        $transactions = $this->db->select("
            SELECT
                st.*,
                sc.id as check_id,
                sc.check_number,
                sc.bank_name as check_bank_name,
                sc.due_date as check_due_date,
                sc.status as check_status
            FROM supplier_transactions st
            LEFT JOIN supplier_checks sc ON sc.supplier_transaction_id = st.id
            WHERE st.supplier_id = ?
            ORDER BY st.transaction_date DESC, st.created_at DESC
        ", [$id]);

        // Calculate balance
        $balance = $this->calculateBalance($id);

        $auth = $this->auth;
        $pageTitle = 'Tedarikçi Detay';
        $currentPage = 'suppliers';

        require_once __DIR__ . '/../views/suppliers/view.php';
    }

    /**
     * Bakiye hesaplama
     */
    private function calculateBalance($supplierId)
    {
        $result = $this->db->selectOne("
            SELECT
                SUM(CASE
                    WHEN transaction_type IN ('purchase', 'adjustment') THEN amount
                    WHEN transaction_type IN ('payment', 'refund') THEN -amount
                    ELSE 0
                END) as balance,
                SUM(CASE WHEN transaction_type = 'purchase' THEN amount ELSE 0 END) as total_purchases,
                SUM(CASE WHEN transaction_type = 'payment' THEN amount ELSE 0 END) as total_payments,
                COUNT(*) as transaction_count
            FROM supplier_transactions
            WHERE supplier_id = ?
        ", [$supplierId]);

        return [
            'balance' => $result['balance'] ?? 0,
            'total_purchases' => $result['total_purchases'] ?? 0,
            'total_payments' => $result['total_payments'] ?? 0,
            'transaction_count' => $result['transaction_count'] ?? 0
        ];
    }

    /**
     * Yeni işlem kaydetme
     */
    public function saveTransaction($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/suppliers/view/' . $id);
        }

        // Validate supplier exists
        $supplier = $this->db->selectOne("SELECT id FROM suppliers WHERE id = ?", [$id]);
        if (!$supplier) {
            Helper::setFlash('danger', 'Tedarikçi bulunamadı.');
            Helper::redirect('/suppliers');
        }

        $this->ensureChecksTable();

        // Validate required fields
        $transactionType = $_POST['transaction_type'] ?? '';
        $amount = $_POST['amount'] ?? 0;
        $transactionDate = $_POST['transaction_date'] ?? '';

        if (empty($transactionType) || empty($amount) || empty($transactionDate)) {
            Helper::setFlash('danger', 'İşlem tipi, tutar ve tarih zorunludur.');
            Helper::redirect('/suppliers/view/' . $id);
        }

        if (!in_array($transactionType, ['purchase', 'payment', 'refund', 'adjustment'])) {
            Helper::setFlash('danger', 'Geçersiz işlem tipi.');
            Helper::redirect('/suppliers/view/' . $id);
        }

        $paymentMethod = $_POST['payment_method'] ?? 'cash';
        $checkNumber = trim($_POST['check_number'] ?? '');
        $checkBankName = trim($_POST['check_bank_name'] ?? '');
        $checkDueDate = $_POST['check_due_date'] ?? '';

        if ($transactionType === 'payment' && $paymentMethod === 'check') {
            if ($checkNumber === '' || $checkDueDate === '') {
                Helper::setFlash('danger', 'Cek ile odeme icin cek no ve vade tarihi zorunludur.');
                Helper::redirect('/suppliers/view/' . $id);
            }
        }

        // Prepare transaction data
        $description = $_POST['description'] ?? null;
        if ($transactionType === 'payment' && $paymentMethod === 'check') {
            $description = trim(($description ? $description . ' | ' : '') . 'Odeme Yontemi: Cek');
        }

        $data = [
            'supplier_id' => $id,
            'transaction_type' => $transactionType,
            'amount' => floatval($amount),
            'transaction_date' => $transactionDate,
            'reference_number' => $_POST['reference_number'] ?? null,
            'description' => $description,
            'created_by' => $this->auth->userId()
        ];

        try {
            $this->db->beginTransaction();

            // Insert transaction
            $transactionId = $this->db->insert('supplier_transactions', $data);

            if (!$transactionId) {
                throw new Exception('Islem kaydi olusturulamadi.');
            }

            // Cek odemesi secildiyse ayri cek tablosuna otomatik kaydet
            if ($transactionType === 'payment' && $paymentMethod === 'check') {
                $checkSaved = $this->db->insert('supplier_checks', [
                    'supplier_id' => $id,
                    'supplier_transaction_id' => $transactionId,
                    'check_number' => $checkNumber,
                    'bank_name' => $checkBankName !== '' ? $checkBankName : null,
                    'amount' => floatval($amount),
                    'issue_date' => $transactionDate,
                    'due_date' => $checkDueDate,
                    'status' => 'BEKLEMEDE',
                    'description' => $_POST['description'] ?? null,
                    'created_by' => $this->auth->userId()
                ]);

                if (!$checkSaved) {
                    throw new Exception('Cek kaydi olusturulamadi.');
                }
            }

            $this->db->commit();
            Helper::setFlash('success', 'İşlem başarıyla kaydedildi.');
        } catch (Exception $e) {
            $this->db->rollback();
            Helper::setFlash('danger', 'İşlem kaydedilirken hata oluştu: ' . $e->getMessage());
        }

        Helper::redirect('/suppliers/view/' . $id);
    }

    /**
     * İşlem silme
     */
    public function deleteTransaction($supplierId, $transactionId)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/suppliers/view/' . $supplierId);
        }

        // Verify transaction belongs to supplier
        $transaction = $this->db->selectOne(
            "SELECT id FROM supplier_transactions WHERE id = ? AND supplier_id = ?",
            [$transactionId, $supplierId]
        );

        if (!$transaction) {
            Helper::setFlash('danger', 'İşlem bulunamadı.');
            Helper::redirect('/suppliers/view/' . $supplierId);
        }

        // Delete transaction
        $result = $this->db->delete('supplier_transactions', 'id = ?', [$transactionId]);

        if ($result) {
            Helper::setFlash('success', 'İşlem başarıyla silindi.');
        } else {
            Helper::setFlash('danger', 'İşlem silinemedi.');
        }

        Helper::redirect('/suppliers/view/' . $supplierId);
    }

    /**
     * Tedarikçi kaydetme (POST handler for create)
     */
    public function store()
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/suppliers');
        }

        // Validate required fields
        $supplier_name = $_POST['supplier_name'] ?? '';
        $amount = $_POST['amount'] ?? 0;

        if (empty($supplier_name) || empty($amount)) {
            Helper::setFlash('danger', 'Tedarikçi adı ve tutar zorunludur.');
            Helper::redirect('/suppliers/create');
        }

        // Prepare data
        $data = [
            'supplier_name' => $supplier_name,
            'invoice_number' => $_POST['invoice_number'] ?? null,
            'amount' => floatval($amount),
            'due_date' => !empty($_POST['due_date']) ? $_POST['due_date'] : null,
            'description' => $_POST['description'] ?? null,
            'status' => $_POST['status'] ?? 'pending',
            'payment_date' => !empty($_POST['payment_date']) ? $_POST['payment_date'] : null,
            'created_by' => $this->auth->userId()
        ];

        // Insert to database
        $id = $this->db->insert('suppliers', $data);

        if ($id) {
            Helper::setFlash('success', 'Tedarikçi başarıyla eklendi.');
            Helper::redirect('/suppliers');
        } else {
            Helper::setFlash('danger', 'Tedarikçi eklenirken hata oluştu.');
            Helper::redirect('/suppliers/create');
        }
    }

    /**
     * Tedarikçi güncelleme (POST handler for edit)
     */
    public function update($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/suppliers');
        }

        // Check if supplier exists
        $existing = $this->db->selectOne("SELECT * FROM suppliers WHERE id = ?", [$id]);
        if (!$existing) {
            Helper::setFlash('danger', 'Tedarikçi bulunamadı.');
            Helper::redirect('/suppliers');
        }

        // Validate required fields
        $supplier_name = $_POST['supplier_name'] ?? '';
        $amount = $_POST['amount'] ?? 0;

        if (empty($supplier_name) || empty($amount)) {
            Helper::setFlash('danger', 'Tedarikçi adı ve tutar zorunludur.');
            Helper::redirect('/suppliers/edit/' . $id);
        }

        // Prepare data
        $data = [
            'supplier_name' => $supplier_name,
            'invoice_number' => $_POST['invoice_number'] ?? null,
            'amount' => floatval($amount),
            'due_date' => !empty($_POST['due_date']) ? $_POST['due_date'] : null,
            'description' => $_POST['description'] ?? null,
            'status' => $_POST['status'] ?? 'pending',
            'payment_date' => !empty($_POST['payment_date']) ? $_POST['payment_date'] : null,
        ];

        // Update database
        $result = $this->db->update('suppliers', $data, 'id = ?', [$id]);

        if ($result !== false) {
            Helper::setFlash('success', 'Tedarikçi başarıyla güncellendi.');
            Helper::redirect('/suppliers');
        } else {
            Helper::setFlash('danger', 'Tedarikçi güncellenirken hata oluştu.');
            Helper::redirect('/suppliers/edit/' . $id);
        }
    }

    /**
     * Tedarikçi silme
     */
    public function delete($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/suppliers');
        }

        $result = $this->db->delete('suppliers', 'id = ?', [$id]);

        if ($result) {
            Helper::setFlash('success', 'Tedarikçi başarıyla silindi.');
        } else {
            Helper::setFlash('danger', 'Tedarikçi silinemedi.');
        }

        Helper::redirect('/suppliers');
    }

    private function ensureChecksTable()
    {
        $this->db->query("
            CREATE TABLE IF NOT EXISTS supplier_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                supplier_id INT NOT NULL,
                supplier_transaction_id INT NOT NULL,
                check_number VARCHAR(100) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                issue_date DATE NOT NULL,
                due_date DATE NOT NULL,
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_supplier_transaction (supplier_transaction_id),
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_supplier (supplier_id),
                FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE CASCADE,
                FOREIGN KEY (supplier_transaction_id) REFERENCES supplier_transactions(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");
    }
}
